//! User-provided program settings, taking into account pyproject.toml and
//! command-line options. Structure mirrors the user-facing representation of
//! the various parameters.

use std::path::{Path, PathBuf};

use anyhow::{anyhow, Result};
use glob::{glob, GlobError, Paths, PatternError};
use regex::Regex;

use crate::checks_gen::CheckCodePrefix;
use crate::cli::{collect_per_file_ignores, Overrides};
use crate::settings::options::Options;
use crate::settings::pyproject::load_options;
use crate::settings::types::{FilePattern, PerFileIgnore, PythonVersion, SerializationFormat};
use crate::{
    flake8_annotations, flake8_bugbear, flake8_import_conventions, flake8_quotes,
    flake8_tidy_imports, fs, isort, mccabe, pep8_naming, pyupgrade,
};

#[derive(Debug, Default)]
pub struct Configuration {
    pub allowed_confusables: Option<Vec<char>>,
    pub dummy_variable_rgx: Option<Regex>,
    pub exclude: Option<Vec<FilePattern>>,
    pub extend: Option<PathBuf>,
    pub extend_exclude: Vec<FilePattern>,
    pub extend_ignore: Vec<Vec<CheckCodePrefix>>,
    pub extend_select: Vec<Vec<CheckCodePrefix>>,
    pub external: Option<Vec<String>>,
    pub fix: Option<bool>,
    pub fixable: Option<Vec<CheckCodePrefix>>,
    pub format: Option<SerializationFormat>,
    pub ignore: Option<Vec<CheckCodePrefix>>,
    pub ignore_init_module_imports: Option<bool>,
    pub line_length: Option<usize>,
    pub per_file_ignores: Option<Vec<PerFileIgnore>>,
    pub respect_gitignore: Option<bool>,
    pub select: Option<Vec<CheckCodePrefix>>,
    pub show_source: Option<bool>,
    pub src: Option<Vec<PathBuf>>,
    pub target_version: Option<PythonVersion>,
    pub unfixable: Option<Vec<CheckCodePrefix>>,
    // Plugins
    pub flake8_annotations: Option<flake8_annotations::settings::Options>,
    pub flake8_bugbear: Option<flake8_bugbear::settings::Options>,
    pub flake8_import_conventions: Option<flake8_import_conventions::settings::Options>,
    pub flake8_quotes: Option<flake8_quotes::settings::Options>,
    pub flake8_tidy_imports: Option<flake8_tidy_imports::settings::Options>,
    pub isort: Option<isort::settings::Options>,
    pub mccabe: Option<mccabe::settings::Options>,
    pub pep8_naming: Option<pep8_naming::settings::Options>,
    pub pyupgrade: Option<pyupgrade::settings::Options>,
}

impl Configuration {
    pub fn from_pyproject(pyproject: &Path, project_root: &Path) -> Result<Self> {
        Self::from_options(load_options(pyproject)?, project_root)
    }

    pub fn from_options(options: Options, project_root: &Path) -> Result<Self> {
        Ok(Configuration {
            allowed_confusables: options.allowed_confusables,
            dummy_variable_rgx: options
                .dummy_variable_rgx
                .map(|pattern| Regex::new(&pattern))
                .transpose()
                .map_err(|e| anyhow!("Invalid `dummy-variable-rgx` value: {e}"))?,
            exclude: options.exclude.map(|paths| {
                paths
                    .into_iter()
                    .map(|pattern| {
                        let absolute = fs::normalize_path_to(Path::new(&pattern), project_root);
                        FilePattern::User(pattern, absolute)
                    })
                    .collect()
            }),
            extend: options.extend.map(PathBuf::from),
            extend_exclude: options
                .extend_exclude
                .map(|paths| {
                    paths
                        .into_iter()
                        .map(|pattern| {
                            let absolute = fs::normalize_path_to(Path::new(&pattern), project_root);
                            FilePattern::User(pattern, absolute)
                        })
                        .collect()
                })
                .unwrap_or_default(),
            extend_ignore: vec![options.extend_ignore.unwrap_or_default()],
            extend_select: vec![options.extend_select.unwrap_or_default()],
            external: options.external,
            fix: options.fix,
            fixable: options.fixable,
            format: options.format,
            ignore: options.ignore,
            ignore_init_module_imports: options.ignore_init_module_imports,
            line_length: options.line_length,
            per_file_ignores: options.per_file_ignores.map(|per_file_ignores| {
                per_file_ignores
                    .into_iter()
                    .map(|(pattern, prefixes)| {
                        let absolute = fs::normalize_path_to(Path::new(&pattern), project_root);
                        PerFileIgnore::new(pattern, absolute, &prefixes)
                    })
                    .collect()
            }),
            respect_gitignore: options.respect_gitignore,
            select: options.select,
            show_source: options.show_source,
            src: options
                .src
                .map(|src| resolve_src(&src, project_root))
                .transpose()?,
            target_version: options.target_version,
            unfixable: options.unfixable,
            // Plugins
            flake8_annotations: options.flake8_annotations,
            flake8_bugbear: options.flake8_bugbear,
            flake8_import_conventions: options.flake8_import_conventions,
            flake8_quotes: options.flake8_quotes,
            flake8_tidy_imports: options.flake8_tidy_imports,
            isort: options.isort,
            mccabe: options.mccabe,
            pep8_naming: options.pep8_naming,
            pyupgrade: options.pyupgrade,
        })
    }

    #[must_use]
    pub fn combine(self, config: Configuration) -> Self {
        Self {
            allowed_confusables: self.allowed_confusables.or(config.allowed_confusables),
            dummy_variable_rgx: self.dummy_variable_rgx.or(config.dummy_variable_rgx),
            exclude: self.exclude.or(config.exclude),
            respect_gitignore: self.respect_gitignore.or(config.respect_gitignore),
            extend: self.extend.or(config.extend),
            extend_exclude: config
                .extend_exclude
                .into_iter()
                .chain(self.extend_exclude.into_iter())
                .collect(),
            extend_ignore: config
                .extend_ignore
                .into_iter()
                .chain(self.extend_ignore.into_iter())
                .collect(),
            extend_select: config
                .extend_select
                .into_iter()
                .chain(self.extend_select.into_iter())
                .collect(),
            external: self.external.or(config.external),
            fix: self.fix.or(config.fix),
            fixable: self.fixable.or(config.fixable),
            format: self.format.or(config.format),
            ignore: self.ignore.or(config.ignore),
            ignore_init_module_imports: self
                .ignore_init_module_imports
                .or(config.ignore_init_module_imports),
            line_length: self.line_length.or(config.line_length),
            per_file_ignores: self.per_file_ignores.or(config.per_file_ignores),
            select: self.select.or(config.select),
            show_source: self.show_source.or(config.show_source),
            src: self.src.or(config.src),
            target_version: self.target_version.or(config.target_version),
            unfixable: self.unfixable.or(config.unfixable),
            // Plugins
            flake8_annotations: self.flake8_annotations.or(config.flake8_annotations),
            flake8_bugbear: self.flake8_bugbear.or(config.flake8_bugbear),
            flake8_import_conventions: self
                .flake8_import_conventions
                .or(config.flake8_import_conventions),
            flake8_quotes: self.flake8_quotes.or(config.flake8_quotes),
            flake8_tidy_imports: self.flake8_tidy_imports.or(config.flake8_tidy_imports),
            isort: self.isort.or(config.isort),
            mccabe: self.mccabe.or(config.mccabe),
            pep8_naming: self.pep8_naming.or(config.pep8_naming),
            pyupgrade: self.pyupgrade.or(config.pyupgrade),
        }
    }

    pub fn apply(&mut self, overrides: Overrides) {
        if let Some(dummy_variable_rgx) = overrides.dummy_variable_rgx {
            self.dummy_variable_rgx = Some(dummy_variable_rgx);
        }
        if let Some(exclude) = overrides.exclude {
            self.exclude = Some(exclude);
        }
        if let Some(extend_exclude) = overrides.extend_exclude {
            self.extend_exclude.extend(extend_exclude);
        }
        if let Some(fix) = overrides.fix {
            self.fix = Some(fix);
        }
        if let Some(fixable) = overrides.fixable {
            self.fixable = Some(fixable);
        }
        if let Some(format) = overrides.format {
            self.format = Some(format);
        }
        if let Some(ignore) = overrides.ignore {
            self.ignore = Some(ignore);
        }
        if let Some(line_length) = overrides.line_length {
            self.line_length = Some(line_length);
        }
        if let Some(max_complexity) = overrides.max_complexity {
            self.mccabe = Some(mccabe::settings::Options {
                max_complexity: Some(max_complexity),
            });
        }
        if let Some(per_file_ignores) = overrides.per_file_ignores {
            self.per_file_ignores = Some(collect_per_file_ignores(per_file_ignores));
        }
        if let Some(respect_gitignore) = overrides.respect_gitignore {
            self.respect_gitignore = Some(respect_gitignore);
        }
        if let Some(select) = overrides.select {
            self.select = Some(select);
        }
        if let Some(show_source) = overrides.show_source {
            self.show_source = Some(show_source);
        }
        if let Some(target_version) = overrides.target_version {
            self.target_version = Some(target_version);
        }
        if let Some(unfixable) = overrides.unfixable {
            self.unfixable = Some(unfixable);
        }
        // Special-case: `extend_ignore` and `extend_select` are parallel arrays, so
        // push an empty array if only one of the two is provided.
        match (overrides.extend_ignore, overrides.extend_select) {
            (Some(extend_ignore), Some(extend_select)) => {
                self.extend_ignore.push(extend_ignore);
                self.extend_select.push(extend_select);
            }
            (Some(extend_ignore), None) => {
                self.extend_ignore.push(extend_ignore);
                self.extend_select.push(Vec::new());
            }
            (None, Some(extend_select)) => {
                self.extend_ignore.push(Vec::new());
                self.extend_select.push(extend_select);
            }
            (None, None) => {}
        }
    }
}

/// Given a list of source paths, which could include glob patterns, resolve the
/// matching paths.
pub fn resolve_src(src: &[String], project_root: &Path) -> Result<Vec<PathBuf>> {
    let globs = src
        .iter()
        .map(Path::new)
        .map(|path| fs::normalize_path_to(path, project_root))
        .map(|path| glob(&path.to_string_lossy()))
        .collect::<Result<Vec<Paths>, PatternError>>()?;
    let paths: Vec<PathBuf> = globs
        .into_iter()
        .flatten()
        .collect::<Result<Vec<PathBuf>, GlobError>>()?;
    Ok(paths)
}
